%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% >>>> IMAGEBOX >>>> JX >>>> UCLA >>>>
%%
%% curvature.m
%%
%% function [curvu, normgrad, dxu, dyu] = curvature(u, difftype, epsilon)
%%
%% get the curvature of u
%% 1D or 2D. h = dx = dy = 1, Neumann BC
%% 
%% curvu = div( grad u / sqrt{|grad u|^2+epsilon^2} )
%% 
%% input:
%%     u:        1D vector or 2D matrix
%%     difftype: 1 = 'D-D+'; -1 = 'D+D-'; 0 = 'D0D0'; default: 1
%%     epsilon:  tiny value number for approximating 
%%               |grad u| ~= sqrt{|grad u|^2+epsilon^2}, default = 1.0e-3
%% 
%% output: 
%%     curvu:    curvature
%%     normgrad: |grad(u)|
%%     dxu:      u_x 
%%     dyu:      u_y (2D only) 
%%
%% created: 03/24/2005
%% last modified: 04/26/2005
%% author: jjxu@ucla
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [curvu, normgrad, dxu, dyu] = curvature(u, difftype, epsilon)

%% check input data

%% epsilon: regularization parameter
if exist('epsilon') ~= 1 | ~isnumeric(epsilon) | epsilon < 0,
    epsilon = 1.0e-3;  %% default value
end;

%% differential type
if exist('difftype') == 1,
    if isnumeric(difftype), 
        difftype = sign(difftype); 
    elseif ischar(difftype)
        if     difftype == 'D-D+', difftype = 1;
        elseif difftype == 'D+D-', difftype = -1;
        else   difftype = 0;
        end
    else 
        difftype = 1;
    end
else 
    difftype = 1;      %% default: D+D-
end

do_trans = (size(u,1) == 1);
if do_trans, u = u(:); end

dim_1d = (size(u,2) == 1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if dim_1d, %% 1D vector
    m   = length(u);
    dxu = zeros(m,1);

    %% gradient part. Neumann BC
    switch difftype
        case 1   %% D+
            dxu(1:m-1) = u(2:m) - u(1:m-1);
            %% dxu(m) = u(m+1) - u(m) = 0; % Neumann BC
        case -1  %% D-
            dxu(2:m)   = u(2:m) - u(1:m-1);
            %% dxu(1) = u(1) - u(0) = 0;   % Neumann BC
        case 0   %% D0
            dxu(2:m-1) = u(3:m) - u(1:m-2);
            dxu(1)     = u(2) - u(1);   %% u(0) = u(1)
            dxu(m)     = u(m) - u(m-1); %% u(m+1) = u(m)
            dxu = 0.5 * dxu;
    end

    %% normalization
    normgrad  = sqrt(dxu.*dxu + epsilon*epsilon);
    normed_du = dxu ./ normgrad;

    %% divergence part. Neumann BC
    curvu = zeros(m,1);

    switch difftype
        case 1   %% D-
            curvu(2:m) = normed_du(2:m) - normed_du(1:m-1);
            curvu(1)   = normed_du(1);    %% du(0) = u(1) - u(0) = 0
        case -1  %% D+
            curvu(1:m-1) = normed_du(2:m) - normed_du(1:m-1);
            curvu(m)     = - normed_du(m); %% du(m+1) = u(m+1) - u(m) = 0
        case 0   %% D0
            curvu(2:m-1) = normed_du(3:m) - normed_du(1:m-2);
            curvu(1)     = normed_du(2) + normed_du(1); 
                    %% du(0) = u(1)-u(-1) = u(1)-u(2) = u(0)-u(2) = -du(1)
            curvu(m)     = -normed_du(m) - normed_du(m-1); 
                    %% du(m+1) = u(m+2)-u(m) = u(m-1)-u(m) = u(m-1)-u(m+1) = -du(m)
            curvu = 0.5 * curvu;
    end

    if do_trans, curvu = curvu.'; normgrad = normgrad.'; dxu = dxu.'; end;
    return;  %% end of 1d.
%%%%
else  %% 2D matrix
    [m,n] = size(u);
    dxu = zeros(m,n); dyu = zeros(m,n);

    switch difftype
        case 1  %% D+
            dxu(1:m-1,:) = u(2:m,:) - u(1:m-1,:);
            dyu(:,1:n-1) = u(:,2:n) - u(:,1:n-1);
            %% dxu(m,:) = 0; dyu(:,n) = 0; %% Neumann BC
        case -1 %% D-
            dxu(2:m,:) = u(2:m,:) - u(1:m-1,:);
            dyu(:,2:n) = u(:,2:n) - u(:,1:n-1);
        case 0  %% D0
            dxu(2:m-1,:) = u(3:m,:) - u(1:m-2,:);
            dxu(1,:)     = u(2,:) - u(1,:);    %% u(0,:) = u(1,:)
            dxu(m,:)     = u(m,:) - u(m-1,:);  %% u(m+1,:) = u(m,:)
            dyu(:,2:n-1) = u(:,3:n) - u(:,1:n-2);
            dyu(:,1)     = u(:,2) - u(:,1);
            dyu(:,n)     = u(:,n) - u(:,n-1);
            dxu = 0.5 * dxu; 
            dyu = 0.5 * dyu;
    end

    normgrad = sqrt(dxu.*dxu + dyu.*dyu + epsilon*epsilon);

    normed_dxu = dxu ./ normgrad;
    normed_dyu = dyu ./ normgrad;

    %% divergence part. Neumann BC
    curvu = zeros(m,n);

    switch difftype
        case 1  %% D-
            curvu(2:m,:) = normed_dxu(2:m,:) - normed_dxu(1:m-1,:);
            curvu(1,:)   = normed_dxu(1,:);      
                         %% dxu(0) = u(1)-u(0) = 0
            curvu(:,2:n) = curvu(:,2:n) + normed_dyu(:,2:n) - normed_dyu(:,1:n-1);
            curvu(:,1)   = curvu(:,1) + normed_dyu(:,1);
                         %% dyu(:,0) = u(:,1)-u(:,0) = 0
        case -1 %% D+
            curvu(1:m-1,:) = normed_dxu(2:m,:) - normed_dxu(1:m-1,:);
            curvu(m,:)     = - normed_dxu(m,:);  
                         %% dxu(m+1) = u(m+1)-u(m) = 0
            curvu(:,1:n-1) = curvu(:,1:n-1) + normed_dyu(:,2:n) - normed_dyu(:,1:n-1);
            curvu(:,n)     = curvu(:,n) - normed_dyu(:,n);
                         %% dyu(:,n+1) = u(:,n+1)-u(:,n) = 0
        case 0  %% D0
            curvu(2:m-1,:) = normed_dxu(3:m,:) - normed_dxu(1:m-2,:);
            curvu(1,:)     = normed_dxu(2,:) + normed_dxu(1,:); 
                         %% 2*dxu(0,:) = u(1,:) - u(-1,:) = u(1,:) - u(2,:) = - 2*dxu(1,:)
            curvu(m,:)     = - normed_dxu(m,:) - normed_dxu(m-1,:);
                         %% 2*dxu(m+1,:) = u(m+2,:) - u(m,:) = u(m-1,:) - u(m,:) = - 2*dxu(m,:)
            curvu(:,2:n-1) = curvu(:,2:n-1) + normed_dyu(:,3:n) - normed_dyu(:,1:n-2);
            curvu(:,1)     = curvu(:,1) + normed_dyu(:,2) + normed_dyu(:,1);
            curvu(:,n)     = curvu(:,n) - normed_dyu(:,n) - normed_dyu(:,n-1);
            curvu = 0.5 * curvu;
    end
    return; %% end of 2d. end
end;

return; 

%% end of file 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
