#include <cstdlib>

#include <iostream>
#include <vector>
#include <array>
#include <map>

#include "include/utilities/cartesiangrid.hh"
#include "include/turing_rhsfunctions.hh"

#include "include/utilities/stopwatch.hh"
#include "include/turing.hh"

/**
 * Executes the simulation.
 */
int main (int argc, char** argv)
{
  // a map for collecting the options
  std::map<std::string,std::string> options;

  // set default parameters
  options["L"] = "100";     // diameter of \Omega in each direction
  options["D"] = "128";  // diameter in cells (D = 1<<7 = 2^7 = 128, -> bitshift)
  options["o"] = "10";  // time step size for data output
  options["T"] = "100";  // maximum time

  // parse command line parameters
  for (int i = 1; i < argc; i++)
  {
    std::string key = argv[i];
    const std::string::size_type sub = key.find('=');
    if (sub == std::string::npos)
      throw std::runtime_error("Failed to parse option " + key);
    std::string val = key.substr(sub+1);
    key.erase(sub);
    options[key] = val;
  }

  // read parameters
  const double L = std::stod(options["L"]);
  const std::size_t D = std::stoul(options["D"]);
  const double o = std::stod(options["o"]);
  const double T = std::stod(options["T"]);

  // create grid
  CartesianGrid<2> grid({D,D},L/D);
  std::cout << "Created a cartesian grid with " << grid.size() << " cells"
            << std::endl;

  // vector and matrix type
  using DiffusionVector = std::vector<double>;
  using DiffusionMatrix = std::vector<DiffusionVector>; // doesn't really matter...

  // fixed-size vector and matrix type for cell-wise coupled reaction problems
  using ReactionVector = std::array<double,2>;
  using ReactionMatrix = std::array<ReactionVector,2>;

  // create spatially discretized right hand side of the diffusion problem,
  // seperated in both components a and b
  const double diff_const_a = 1.0;
  const double diff_const_b = 10.0;
  DiffusionRhs<DiffusionVector, DiffusionMatrix> diffusionRhs_a(grid, diff_const_a);
  DiffusionRhs<DiffusionVector, DiffusionMatrix> diffusionRhs_b(grid, diff_const_b);

  // create spatially discretized right hand side of the reaction problem
  ReactionRhs<ReactionVector,ReactionMatrix> reactionRhs(2.2, 0.02, 0.0007, 1.1, 0.0002);

  // choose time step size according to CFL-condition
  const double dt = grid.meshwidth() * grid.meshwidth() / (4.0 * std::max(diff_const_a,diff_const_b));

  Turing turing(grid, T, o);

  turing.simulate(diffusionRhs_a, diffusionRhs_b, reactionRhs, dt);

  // exit without an error
  return 0;
}
