#ifndef TURING_VTKWRITER_HH
#define TURING_VTKWRITER_HH

#include <cstdio>

#include <string>
#include <vector>
#include <stdexcept>
#include <sstream>
#include <iomanip>

#include "cartesiangrid.hh"


/**
 * A class for data output in our turing model code.
 * Data is written in the VTK file format which is readable
 * e.g. by ParaView (www.paraview.org).
 */
class Turing_VTKWriter
{
public:
  /**
   * Creates an instance of a VTK writer.
   * @param[in] basename The basic filename for data output.
   * @param[in] grid     Grid which is used in the simulation.
   * @param[in] data     Vector which contains the solution's
   *                     function value in each grid cell.
   */
  Turing_VTKWriter (std::string basename,
                    const CartesianGrid<2>& grid,
                    const std::vector<double>& data)
    : count_(0), basename_(std::move(basename)), grid_(grid), data_(data)
  {}

  /**
   * Write a single time step of the simulated turing model.
   */
  void write ()
  {
    // create filename
    std::stringstream name_builder;
    name_builder << basename_ << "-" << std::setw(6) << std::setfill('0') << count_ << ".vtk";
    std::string name = name_builder.str();

    // create file
    std::ofstream out(name);
    if (!out)
      throw std::runtime_error(std::string("Failed to open file ") + name);

    // write header
    out << "# vtk DataFile Version 2.0\n"
           "vtk output\n"
           "ASCII\n"
           " \n"
           " \n"
           "DATASET STRUCTURED_POINTS\n"
           "DIMENSIONS " << grid_.size(0)+1 << " " << grid_.size(1)+1 << " 1\n"
        << "ORIGIN 0.0 0.0 0.0\n"
           "SPACING " << grid_.meshwidth() << " " << grid_.meshwidth() << " " << grid_.meshwidth() << "\n"
        << "\n"
           "CELL_DATA " <<  grid_.size() << "\n"
        << "SCALARS " << basename_ << " float 1\n"
        << "LOOKUP_TABLE default\n";

    // write data
    for (std::size_t iy = 0; iy < grid_.size(1); iy++)
      for (std::size_t ix = 0; ix < grid_.size(0); ix++)
        out << data_[grid_.index({ix, iy})] << std::endl;

    // update timestep
    count_ += 1;
  }

private:
  unsigned int count_;
  const std::string basename_;
  const CartesianGrid<2>& grid_;
  const std::vector<double>& data_;
};


#endif // TURING_VTKWRITER_HH
