#ifndef TURING_RESERVEDVECTOR_HH
#define TURING_RESERVEDVECTOR_HH

#include <array>
#include <cassert>
#include <algorithm>


/**
 * class which behaves like std::array, with variable size
 * @tparam T data type
 * @tparam N maximal number of elements
 */
template<typename T, std::size_t N>
class ReservedVector{
public:
  ReservedVector() = default;

  ReservedVector(const std::initializer_list<T>& l) : size(l.size()) {
    assert((size <= N));
    std::copy(l.begin(), l.end(), data.begin());
  }

  template<std::size_t M>
  ReservedVector(std::array<T, M> o) : size(M) {
    assert((size <= N));
    std::copy(o.begin(), o.end(), data.begin());
  }

  template<std::size_t M>
  operator std::array<T, M>() {
    assert((size <= M));
    std::array<T, M> o = {};
    std::copy(this->cbegin(), this->cend(), o.begin());
    return o;
  }

  void push_back(const T& v){
    assert((size < N));
    data[size++] = v;
  }

  decltype(auto) begin(){
    return data.begin();
  }
  decltype(auto) begin() const{
    return data.begin();
  }
  decltype(auto) cbegin() const{
    return data.cbegin();
  }

  decltype(auto) end(){
    return this->begin() + size;
  }
  decltype(auto) end() const{
    return this->begin() + size;
  }
  decltype(auto) cend() const {
    return this->cbegin() + size;
  }

private:
  std::array<T, N> data = {};
  std::size_t size = 0;
};

#endif //TURING_RESERVEDVECTOR_HH
