#ifndef TURING_CARTESIANGRID_HH
#define TURING_CARTESIANGRID_HH

#include <cassert>

#include <vector>
#include <array>
#include <numeric>
#include <utility>
#include <mpi.h>
#include "reservedvector.hh"


/**
 * A class for generating informations about a Cartesian grid
 * which can be used in a cell centered finite volume scheme.
 */
template<std::size_t D>
class CartesianGrid
{
public:
  /**
   * Creates an instance of a Cartesian d-dimensional grid.
   * @param[in] x Number of cells in x direction.
   * @param[in] y Number of cells in y direction.
   * @param[in] h Size of the grid cells' edges.
   */
  CartesianGrid (const std::array<std::size_t,D> N, const double h)
      : N_(N), h_(h) {}

  CartesianGrid() = default;

  /**
   * Returns the total number of grid cells.
   */
  [[nodiscard]] std::size_t size () const
  {
    return std::accumulate(N_.begin(), N_.end(), 1., std::multiplies<>());
  }

  /**
   * Returns the number of grid cells in direction d.
   */
  [[nodiscard]] std::size_t size (const std::size_t d) const
  {
    assert(d < D);
    return N_[d];
  }

  /**
   * Returns the mesh width, i.e. the size of the grid cells' edges.
   */
  [[nodiscard]] double meshwidth () const
  {
    return h_;
  }

  /**
   * Returns the index of a cell based on the cell's multi-index i;
   * this induces a linear ordering of the grid cells which can be
   * used to assign data to the grid like a piecewise constant grid
   * function, i.e. a function with a constant value on each cell.
   */
  std::size_t index (const std::array<std::size_t, D>& i) const
  {
    std::size_t index = 0;
    std::size_t offset = 1;
    for(std::size_t d = 0; d < D; ++d)
    {
      index += i[d]*offset;
      offset *= N_[d];
    }
    return index;
  }

  /**
   * Returns the linear index of the neighboring cells
   * @param i multiindex of the current cell
   */
  ReservedVector<std::size_t, 2 * D>  neighbours (const std::array<std::size_t, D>& i) const
  {
    // copy index of the cell we are interested in
    ReservedVector<std::size_t, 2 * D> n;

    // get neighbors in each direction d
    for (std::size_t d = 0; d < D; d++)
    {
      // check left (with respect to direction d)
      if (i[d] > 0){
        n.push_back(index(shift(i, d, -1)));
      }
      // check right (with respect to direction d)
      if (i[d] < N_[d] - 1){
        n.push_back(index(shift(i, d, 1)));
      }
    }
    return n;
  }

private:
  std::array<std::size_t, D> shift(const std::array<std::size_t, D>& i, const std::size_t d, const int s) const{
    auto j = i;
    j[d] += s;
    return j;
  }

  std::array<std::size_t, D> N_ = {};
  double h_ = {};
};


template<std::size_t D>
class ParallelCartesianGrid : public CartesianGrid<D>{
  using Base = CartesianGrid<D>;

public:
  // eventuell weitere Parameter
  ParallelCartesianGrid(const std::array<std::size_t, D> N, const double h, MPI_Comm comm)
  : Base(N, h), grid_comm(comm) {}


  // communicate (send & recv) data in buf with all neighbor grids
  void communicate(std::vector<double>& buf) = delete;

private:
  MPI_Comm grid_comm;

  // eventuell weitere Funktionen, oder Daten
  // z.B. indizes der owner cells / ghost cells
};

#endif // TURING_CARTESIANGRID_HH
