#ifndef TURING_RHSFUNCTIONS_HH
#define TURING_RHSFUNCTIONS_HH

#include <cassert>

#include <vector>

#include "interfaces/timefunction.hh"
#include "utilities/cartesiangrid.hh"


// ******************************* function declarations *******************************
// here, the spatially discretized right hand sides for both splitting parts are defined
// *************************************************************************************


/**
 * Right hand side for each component of the diffusion problem which is discretized
 * using the cell centered finite volume scheme.
 */
template <typename Vector, typename Matrix>
class DiffusionRhs: public DifferentiableTimeFunction<Vector,Vector,Matrix>
{
public:
  using DomainType = typename DiffusionRhs::DomainType;
  using RangeType = typename DiffusionRhs::RangeType;
  using TimeType = typename DiffusionRhs::TimeType;
  using JacobianRangeType = typename DiffusionRhs::JacobianRangeType;

  DiffusionRhs (CartesianGrid<2>& grid_, const double diff_const_)
      : grid(grid_), diff_const(diff_const_)
  {}

  // evaluate discrete function using all neighbours in the grid
  RangeType operator()(const TimeType /*t*/, const DomainType& x) const override
  {
    // dimensions must agree
    RangeType f(x.size());

    // loop over all grid cells and approximate diffusion operator in each cell
    for (std::size_t j = 0; j < grid.size(0); j++)
      for (std::size_t k = 0; k < grid.size(1); k++)
      {
        const auto index = grid.index({j, k});     // get new index from multi-index
        const auto neighbours = grid.neighbours({j, k});  // calculate neighbours of current cell
        f[index] = 0.0;                                    // prepare result

        // approximate Laplace operator with cell centered finite volume scheme
        for(const auto neighbour: neighbours)
          f[index] += x[neighbour] - x[index];
        f[index] = f[index] * diff_const / (grid.meshwidth() * grid.meshwidth());
      }
    return f;
  }

  // not necessary for explicit Euler time-stepping, but has to be implemented anyway
  JacobianRangeType evaluateJacobian (const TimeType /*t*/,
                                      const DomainType& /*x*/) const override
  {
    // assert that we never reach this part of the code
    throw std::runtime_error("DiffusionRhs::evaluateJacobian() should never be called!");
  }

private:
  CartesianGrid<2>& grid;
  double diff_const;
};


/**
 * Right hand side for the reaction problem which is discretized using the cell
 * centered finite volume scheme.
 * The function uses the reaction kinetics g_1(a,b) and g_2(a,b) of the
 * Belousov-Zhabotinsky reaction presented in [Bansagi et al., 2011].
 */
template <typename Vector, typename Matrix>
class ReactionRhs: public DifferentiableTimeFunction<Vector,Vector,Matrix>{
public:
  using DomainType = typename ReactionRhs::DomainType;
  using RangeType = typename ReactionRhs::RangeType;
  using TimeType = typename ReactionRhs::TimeType;
  using JacobianRangeType = typename ReactionRhs::JacobianRangeType;

  ReactionRhs (const double epsilon0, const double epsilon1,
               const double m, const double p, const double q)
      : epsilon0_(epsilon0), epsilon1_(epsilon1), m_(m), p_(p), q_(q)
  {}

  RangeType operator()(const TimeType /*t*/, const DomainType& x) const override
  {
    RangeType f = {};
    f[0] = 1 / epsilon0_ * (w_0(x[1]) * x[0] + w_1(x[0]) * x[1] - x[0] * x[0]);
    f[1] = w_0(x[1]) * x[0] - x[1];
    return f;
  }

  JacobianRangeType evaluateJacobian (const TimeType /*t*/, const DomainType& x) const override
  {
    JacobianRangeType df = {};
    df[0][0] = (w_0(x[1]) - (2 * p_ * q_ * x[1]) / ((q_ + x[0]) * (q_ + x[0])) - 2 * x[0]) / epsilon0_;
    df[0][1] = (-epsilon1_ * m_ / (1 - m_ * x[1] + epsilon1_) /
                (1 - m_ * x[1] + epsilon1_) * x[0] + w_1(x[0])) / epsilon0_;
    df[1][0] = w_0(x[1]);
    df[1][1] = -epsilon1_ * m_ / (1 - m_ * x[1] + epsilon1_) / (1 - m_ * x[1] + epsilon1_) * x[0] - 1;
    return df;
  }

private:
  [[nodiscard]] double w_0 (double b) const
  {
    return (1.0 - m_ * b) / (1.0 - m_ * b + epsilon1_);
  }

  [[nodiscard]] double w_1 (double a) const
  {
    return p_ * (q_ - a) / (q_ + a);
  }

  double epsilon0_;
  double epsilon1_;
  double m_;
  double p_;
  double q_;
};


#endif // TURING_RHSFUNCTIONS_HH
