#include <cstdlib>

#include <iostream>
#include <vector>
#include <array>
#include <map>
#include <fstream>
#include <random>

#include "utilities/cartesiangrid.hh"
#include "utilities/vtkwriter.hh"
#include "turing_rhsfunctions.hh"
#include "newtonsolver.hh"
#include "thetascheme.hh"

#include "utilities/stopwatch.hh"


class Turing{
public:
  Turing(const CartesianGrid<2>& grid_, const double T_, const double o_) : grid(grid_), T(T_), o(o_) {}

  template<typename DiffusionRhs, typename ReactionRhs>
  void simulate(const DiffusionRhs& diff_a, const DiffusionRhs& diff_b, const ReactionRhs& r, const double dt){
    using DiffusionVector = typename DiffusionRhs::DomainType;
    using DiffusionMatrix = typename DiffusionRhs::JacobianRangeType;
    using ReactionVector = typename ReactionRhs::DomainType;
    using ReactionMatrix = typename ReactionRhs::JacobianRangeType;

    // allocate memory for data
    DiffusionVector a(grid.size());           // input of explicit euler
    DiffusionVector b(grid.size());           // input of explicit euler

    ReactionVector c;                // input of implicit euler

    // create VTK writers
    Turing_VTKWriter vtkWriter_a("output/turing_a", grid, a);
    Turing_VTKWriter vtkWriter_b("output/turing_b", grid, b);

    // set initial values
    {
      std::random_device rd;
      std::mt19937 gen(rd());
      std::uniform_real_distribution<> dis(0., 1e-2);

      std::generate(begin(a), end(a), [&gen, &dis](){return dis(gen);});
      std::fill(begin(b), end(b), 0.);
    }

    // output initial values to VTK file
    vtkWriter_a.write();
    vtkWriter_b.write();

    // create and open file for total masses of a and b, usable for visualization
    // with gnuplot
    std::ofstream total_mass;
    total_mass.open("output/total_mass.data");

    // compute total masses at time t+dt
    {
      const double mass_a = std::accumulate(cbegin(a), cend(a), 0.);  // total mass of substance a
      const double mass_b = std::accumulate(cbegin(b), cend(b), 0.);  // total mass of substance b

      // output initial total masses to file
      std::cout << "write total mass -- time: " << 0.0 << std::endl;
      total_mass << 0.0 << " " << mass_a << " " << mass_b << " " << mass_a + mass_b << std::endl;
    }
    // choose Newton's method with threshold 10^-4 as a solver for the
    // reaction problem
    NewtonSolver<ReactionVector,ReactionMatrix> newtonSolver(1e-4, 100);
    NoSolver<DiffusionVector,DiffusionMatrix > noSolver;

    // create a Theta-scheme time integrator with spatially discretized
    // right hand side of the diffusion problem; use explicit Euler method,
    // i.e. Theta == 0.0
    ThetaScheme explicitDiffusionTimeIntegrator_a(diff_a, 0.0, noSolver);
    ThetaScheme explicitDiffusionTimeIntegrator_b(diff_b, 0.0, noSolver);


    // create a Theta-scheme time integrator with spatially discretized
    // right hand side of the reaction problem; use implicit Euler method,
    // i.e. Theta = 1.0
    ThetaScheme implicitReactionTimeIntegrator(r, 1., newtonSolver);

    // prepare time loop
    double t = 0.0;
    double v = o;

    Stopwatch watch;
    watch.start();
    // time loop
    while (t < T)
    {
      // do first explicit Euler step with half step size
      a = explicitDiffusionTimeIntegrator_a.apply(t, dt/2.0, a);
      b = explicitDiffusionTimeIntegrator_b.apply(t, dt/2.0, b);

      // do implicit Euler step for each grid cell with full step size
      for (std::size_t i = 0; i < grid.size(); i++)
      {
        // initialize input
        c[0] = a[i];
        c[1] = b[i];

        // do actual step
        const auto c_next = implicitReactionTimeIntegrator.apply(t, dt, c);

        // update concentrations
        a[i] = c_next[0];
        b[i] = c_next[1];
      }

      // do second explicit Euler step with half step size
      a = explicitDiffusionTimeIntegrator_a.apply(t + dt / 2.0, dt / 2.0, a);
      b = explicitDiffusionTimeIntegrator_b.apply(t + dt / 2.0, dt / 2.0, b);

      // end of Strang Splitting

      // data output, controlled by time step size vt
      if (v <= t + dt)
      {
        std::cout << "write vtk file and total mass -- time: " << t + dt
                  << std::endl;

        // output solution at time t+dt to VTK file
        vtkWriter_a.write();
        vtkWriter_b.write();

        // compute total masses at time t+dt
        const double mass_a = std::accumulate(cbegin(a), cend(a), 0.);  // total mass of substance a
        const double mass_b = std::accumulate(cbegin(b), cend(b), 0.);  // total mass of substance b

        // output total masses at time t+dt to file
        total_mass << t + dt << " " << mass_a << " " << mass_b << " " << mass_a + mass_b << std::endl;

        // prepare next data output
        v += o;
      }

      // prepare next time step
      t += dt;
    }

    watch.stop();
    // output solution at final time to VTK file
    vtkWriter_a.write();
    vtkWriter_b.write();

    // close file which contains total masses
    total_mass.close();
    std::cout << "Elapsed time: " << watch.elapsed() << std::endl;

  }

private:
  CartesianGrid<2> grid = {};
  double T = {};
  double o = {};
};