#ifndef NEWTONSOLVER_HH
#define NEWTONSOLVER_HH

#include <cassert>
#include <cmath>

#include <iostream>
#include <algorithm>
#include <type_traits>

#include "interfaces/solver.hh"


template <typename Vector, typename Matrix>
class NewtonSolver
    : public Solver<Vector,Matrix>
{
public:
  using VectorType = typename NewtonSolver::VectorType;
  using MatrixType = typename NewtonSolver::MatrixType;
  using FunctionType = typename NewtonSolver::FunctionType;

public:
  /**
   * Constructs a new Newton solver.
   * @param[in] eps Threshold for stopping criterion; the solver stops
   *                if rate of change is below eps.
   */
  explicit NewtonSolver (double eps, std::size_t max_iter)
      : eps_(eps), max_iter_(max_iter)
  {
    // print status message
    std::cout << "Created new instance of NewtonSolver with eps = " << eps_
              << " and max_iter = " << max_iter_ << std::endl;
  }

  /**
   * Apply Newton's method, i.e. solve the equation.
   * @param[in]     r Function which defines the equation (residual function).
   * @param[in,out] z Solution of the equation;
   *                  contains initial guess.
   */
  VectorType apply (const FunctionType& r, const VectorType& z) const override
  {
    VectorType y(z);

    // iterate on z at least once (using the initial guess)
    for (std::size_t k = 0; k < max_iter_; ++k) {
      const auto r_value = r(y);
      // evaluate residual function and its jacobian
      const auto r_jacobian = r.evaluateJacobian(y);

      // solve linear system r_jacobian * update = r_value
      const auto r_jacobian_inv = inversion(r_jacobian);
      const auto update = mv_mult(r_jacobian_inv, r_value);

      // update z
      for (unsigned int i = 0; i < z.size(); i++)
        y[i] -= update[i];

      if(two_norm(update) < eps_)
        return y;
    }
    throw std::runtime_error("Newton solver did not converge after " + std::to_string(max_iter_) + " iterations.");
  }

private:
  // inverts a matrix (currently only 1x1, 2x2 and 3x3 matrices are supported)
  MatrixType inversion (const MatrixType& m) const
  {
    MatrixType inv = {};
    const double det = m[0][0]*m[1][1]-m[0][1]*m[1][0];
    assert(det != 0.0);
    inv[0][0] =  m[1][1]/det; inv[0][1] = -m[0][1]/det;
    inv[1][0] = -m[1][0]/det; inv[1][1] =  m[0][0]/det;
    return inv;
  }

  // realizes matrix-vector multiplication
  VectorType mv_mult (const MatrixType& m, const VectorType& v) const
  {
    VectorType result = {};
    std::fill(result.begin(), result.end(), 0.);
    for (unsigned int i = 0; i < m.size(); i++)
      for (unsigned int j = 0; j < v.size(); j++)
      {
        assert(m[i].size() == v.size());

        result[i] += m[i][j] * v[j];
      }
    return result;
  }

  // calculates the Euclidean norm of a vector
  double two_norm (const VectorType& v) const
  {
    double norm_sqr = 0.0;
    for (unsigned int i = 0; i < v.size(); i++)
      norm_sqr += v[i] * v[i];
    return sqrt(norm_sqr);
  }

private:
  const double eps_;
  const std::size_t max_iter_;
};


#endif // NEWTONSOLVER_HH
