#ifndef TIMEFUNCTION_HH
#define TIMEFUNCTION_HH

/**
 * Interface for representing time-dependent, spatially differentiable
 * functions
 * f: R \times R^n --> R^m, (t,x) |--> f(t,x).
 * @tparam Domain        The type representing the function's spatial
 *                       domain R^n.
 * @tparam Range         The type representing the function's range R^m.
 * @tparam JacobianRange The type representing the range R^{m \times n}
 *                       of the function's spatial jacobian.
 * @tparam Time          The type representing the function's temporal
 *                       domain R.
 */
template <typename Domain, typename Range, typename JacobianRange,
          typename Time = double>
class DifferentiableTimeFunction
{
public:
  using DomainType = Domain;
  using RangeType = Range;
  using TimeType = Time;
  using JacobianRangeType = JacobianRange;
  
  /**
   * Pure virtual function (abstract method) that maps t in TimeType
   * and x in DomainType to function value.
   * @param[in]  t  Point in temporal domain.
   * @param[in]  x  Point in spatial domain.
   * @return df Value f(t,x)
   */
  virtual RangeType operator()(const TimeType t, const DomainType& x) const = 0;
  
  /**
   * Pure virtual function (abstract method) that maps t in TimeType
   * and x in DomainType to spatial jacobian matrix Df(t,x) in
   * JacobianRangeType.
   * @param[in]  t  Point in temporal domain.
   * @param[in]  x  Point in spatial domain.
   * @return df Value Df(t,x), i.e. the function's spatial jacobian
   *                matrix in (t,x).
   */
  virtual JacobianRangeType evaluateJacobian (const TimeType t, const DomainType& x) const = 0;
};

#endif // TIMEFUNCTION_HH
