#ifndef SOLVER_HH
#define SOLVER_HH

#include "function.hh"


/**
 * Interface for representing solvers for equations
 *   r(z) = 0,
 * with r: R^n --> R^n differentiable, n \in N.
 * @tparam Vector The type representing a vector in R^n.
 * @tparam Matrix The type representing a matrix in R^{n \times n},
 *                e.g. jacobian matrices of r.
 */
template <typename Vector, typename Matrix>
class Solver
{
public:
  using VectorType = Vector;
  using MatrixType = Matrix;
  using FunctionType = DifferentiableFunction<VectorType,VectorType,MatrixType>;

  /**
   * Pure virtual function (abstract method) that applies the solver,
   * i.e. solves the equation.
   * @param[in] r Function which defines the equation (residual function).
   * @param[in] z Solution of the equation;
   *                  contains initial guess e.g. for iterative solvers.
   * @return the solution vector
   */
  virtual VectorType apply (const FunctionType& r, const VectorType& z) const = 0;
};

template<typename Vector, typename Matrix>
class NoSolver: public Solver<Vector, Matrix>{
public:
  using VectorType = typename NoSolver::VectorType;
  using MatrixType = typename NoSolver::MatrixType;
  using FunctionType = typename NoSolver::FunctionType;

  VectorType apply (const FunctionType&, const VectorType& z) const override { return z; }
};

#endif // SOLVER_HH
