#ifndef FUNCTION_HH
#define FUNCTION_HH

/**
 * Interface for representing differentiable functions
 * f: R^n --> R^m, x |--> f(x).
 * @tparam Domain        The type representing the function's domain R^n.
 * @tparam Range         The type representing the function's range R^m.
 * @tparam JacobianRange The type representing the range R^{m \times n}
 *                       of the function's jacobian.
 */
template <typename Domain, typename Range, typename JacobianRange>
class DifferentiableFunction
{
public:
  using DomainType = Domain;
  using RangeType = Range;
  using JacobianRangeType = JacobianRange;

  /**
   * Pure virtual function (abstract method) that maps
   * x in DomainType to f(x) in RangeType.
   * Overloads () such that an object of this class
   * can act as a function.
   * @param[in]  x Point in domain R^n.
   * @return The function value f(x).
   */
  virtual RangeType operator()(const DomainType& x) const = 0;
  
  /**
   * Pure virtual function (abstract method) that maps x in DomainType
   * to jacobian matrix Df(x) in JacobianRangeType.
   * @param[in]  x  Point in domain R^n.
   * @return Value Df(x), i.e. the function's jacobian matrix in x.
   */
  virtual JacobianRange evaluateJacobian (const DomainType& x) const = 0;
};

#endif // FUNCTION_HH
