#ifndef NBODY_VTKWRITER_HH
#define NBODY_VTKWRITER_HH

#include <cstdio>

#include <iostream>
#include <string>
#include <sstream>
#include <iomanip>

#include "nbody_datawrapper.hh"
#include "nbody_exception.hh"


/**
 * A class for data output in our n-body problem code.
 * Data is written in the VTK file format which is readable
 * e.g. by ParaView (www.paraview.org).
 * @tparam DataVariant Type representing the data set which
 *                     contains the position, velocity and mass
 *                     of the bodies.
 */
template <typename DataVariant>
class NBody_VTKWriter
{
public:
  /**
   * Creates an instance of a VTK writer.
   * @param[in] data     Data set which contains the position,
   *                     velocity and mass of the bodies.
   * @param[in] basename The basic filename for data output.
   */
  NBody_VTKWriter (DataVariant& data, std::string basename)
    : wdata_(data), basename_(std::move(basename))
  {}

  /**
   * Write a single timestep of the nbody problem.
   * @param[in] t  Current simulation time.
   * @param[in] dt Current simulation time step size.
   * @param[in] i  Current file number (for the filename).
   */
  void write (double t, double dt, int i)
  {
    std::stringstream name_builder;
    name_builder << basename_ << "_" << std::setw(6) << std::setfill('0') << i << ".vtk";
    std::string name = name_builder.str();

    std::cout << "writing vtk file " << name << std::endl;

    FILE* file;
    file = fopen(name.c_str(),"w");
    if (!file)
      throw NBody_Exception(std::string("Failed to open file ") + name);
    write_vtk_file_double(file,t,dt);
    fclose(file);
  }

private:
  void write_vtk_file_double (FILE* f, double t, double dt)
  {
    // get number of bodies
    std::size_t num_bodies = wdata_.size();

    // vtk header
    fprintf(f,"%s\n","# vtk DataFile Version 1.0");
    fprintf(f,"NBODY %22.16g %22.16g\n",t,dt);
    fprintf(f,"%s\n","ASCII");

    // write with dataset format "polygonal data"
    fprintf(f,"%s\n","DATASET POLYDATA");
    fprintf(f,"%s %zu %s\n","POINTS",num_bodies,"float");
    for (std::size_t i = 0; i < num_bodies; i++)
      fprintf(f,"%22.16g %22.16g %22.16g\n",
              wdata_.r(i)[0],wdata_.r(i)[1],wdata_.r(i)[2]);

    // write polygonal dataset topology (here only vertices)
    fprintf(f,"%s %zu %zu\n","VERTICES",num_bodies,2*num_bodies);
    for (std::size_t i = 0; i < num_bodies; i++)
      fprintf(f,"%zu %zu\n",1l,i);

    // write dataset attributes (data in points)
    fprintf(f,"%s %zu\n","POINT_DATA",num_bodies);

    // write dataset attributes (data in points): scalar data
    fprintf(f,"%s\n","SCALARS mass float");
    fprintf(f,"%s\n","LOOKUP_TABLE default");
    for (std::size_t i = 0; i < num_bodies; i++)
      fprintf(f,"%22.16g\n",wdata_.m(i));

    // write dataset attributes (data in points): vector data
    fprintf(f,"%s\n","VECTORS velocity float");
    for (std::size_t i = 0; i < num_bodies; i++)
      fprintf(f,"%22.16g %22.16g %22.16g\n",
              wdata_.v(i)[0],wdata_.v(i)[1],wdata_.v(i)[2]);
  }

private:
  // store data set reference (wrapped for uniform data access)
  const NBody_DataWrapper<DataVariant> wdata_;

  // store basic filename
  std::string basename_;
};


#endif // NBODY_VTKWRITER_HH
