#ifndef NBODY_VECTOROPERATIONS_HH
#define NBODY_VECTOROPERATIONS_HH

#include <cmath>
#include "nbody_datastructure.hh"


// ********************** function declarations *********************
// overload the C++ arithmetic operators to support vector operations
// ******************************************************************


/**
 * Computes the sum of two vectors in R^3.
 * @param[in,out] lhs Vector in R^3;
 *                    contains first summand and the result.
 * @param[in]     rhs Vector in R^3.
 * @return Reference to lhs (after summation).
 */
Vector3D& operator+= (Vector3D& lhs, const Vector3D& rhs)
{
  for (std::size_t i = 0; i < lhs.size(); ++i)
    lhs[i] += rhs[i];

  return lhs;
}


/**
 * Computes the difference of two vectors in R^3.
 * @param[in,out] lhs Vector in R^3;
 *                    contains minuend and the result.
 * @param[in]     rhs Vector in R^3.
 * @return Reference to lhs (after subtraction).
 */
Vector3D& operator-= (Vector3D& lhs, const Vector3D& rhs)
{
  for (std::size_t i = 0; i < lhs.size(); ++i)
    lhs[i] -= rhs[i];

  return lhs;
}


/**
 * Computes scalar multiplication in R^3,
 * i.e. the multiplication of a vector in R^3 by a scalar in R.
 * @param[in,out] lhs Vector in R^3;
 *                contains input vector and the result.
 * @param[in]     rhs Scalar in R.
 * @return Reference to lhs (after scalar multiplication).
 */
Vector3D& operator*= (Vector3D& lhs, const double& rhs)
{
  for(auto& x: lhs)
    x *= rhs;

  return lhs;
}


/**
 * Computes the sum of two vectors in R^3.
 * @param[in] lhs Vector in R^3.
 * @param[in] rhs Vector in R^3.
 * @return Vector containing the result lhs+rhs.
 */
Vector3D operator+ (const Vector3D& lhs, const Vector3D& rhs)
{
  // copy left operand
  Vector3D result(lhs);

  // implement via operator+= for Vector3D
  result += rhs;

  return result;
}


/**
 * Computes the difference of two vectors in R^3.
 * @param[in] lhs Vector in R^3.
 * @param[in] rhs Vector in R^3.
 * @return Vector containing the result lhs-rhs.
 */
Vector3D operator- (const Vector3D& lhs, const Vector3D& rhs)
{
  // copy left operand
  Vector3D result(lhs);

  // implement via operator-= for Vector3D
  result -= rhs;

  return result;
}


/**
 * Computes scalar multiplication in R^3,
 * i.e. the multiplication of a vector in R^3 by a scalar in R.
 * @param[in] lhs Scalar in R.
 * @param[in] rhs Vector in R^3.
 * @return Vector containing the result lhs*rhs.
 */
Vector3D operator* (const double& lhs, const Vector3D& rhs)
{
  // copy right operand
  Vector3D result(rhs);

  // implement via operator*= for Vector3D
  result *= lhs;

  return result;
}


/**
 * Computes scalar multiplication in R^3,
 * i.e. the multiplication of a vector in R^3 by a scalar in R.
 * @param[in] lhs Vector in R^3.
 * @param[in] rhs Scalar in R.
 * @return Vector containing the result rhs*lhs.
 */
Vector3D operator* (const Vector3D& lhs, const double& rhs)
{
  // implement via "scalar times vector" operator
  return rhs * lhs;
}


/**
 * Multiplies a vector in R^3 by the multiplicative inverse of a
 * scalar in R.
 * @param[in] lhs Vector in R^3.
 * @param[in] rhs Scalar in R.
 * @return Vector containing the result (1/rhs)*lhs.
 */
Vector3D operator/ (const Vector3D& lhs, const double& rhs)
{
  // implement via "scalar times vector" operator
  return (1.0 / rhs) * lhs;
}

double norm(const Vector3D& v){
  double norm = 0;
  for(const auto x: v)
    norm += x * x;
  return std::sqrt(norm);
}


#endif // NBODY_VECTOROPERATIONS_HH
