#ifndef NBODY_HH
#define NBODY_HH

#include <string>
#include <vector>

#include "utilities/nbody_datastructure.hh"
#include "utilities/nbody_initialsetup.hh"
#include "utilities/nbody_vtkwriter.hh"
#include "utilities/nbody_vectoroperations.hh"
#include "utilities/nbody_stopwatch.hh"
#include "nbody_acceleration.hh"
#include "leapfrog_scheme.hh"


/**
 * A class for n-body simulations.
 */
class NBody
{
public:
  /**
   * Creates an instance of an n-body simulation.
   * @param[in] N    Number of bodies.
   * @param[in] T    Maximum time.
   * @param[in] dt   Time step size.
   * @param[in] o    Time step size for data output.
   * @param[in] init Name of initial configuration.
   */
  NBody (const std::size_t N, const double T, const double dt, const double o,
         std::string init)
      : N_(N), T_(T), dt_(dt), o_(o), init_(std::move(init))
  { }

  /**
   * Executes the simulation.
   */
  void simulate () const
  {
    // choose one of the following the data layout variants
    DataVariant1 data;

    // get initial values from initial setup generator
    NBody_InitialSetup<DataVariant1> initialSetup(data);
    initialSetup.generate(init_, N_);

    // output initial values
    NBody_VTKWriter<DataVariant1> vtkWriter(data,"output/nbody");
    vtkWriter.write(0.0,dt_,0);

    // right hand side function of the n-body initial value problem of second order which computes a
    // vector containing the acceleration of the bodies
    NBody_Acceleration f;
    f.setupTree(data);

    // scheme for numerically solving the n-body initial
    // value problem
    double t_out = o_;
    double t = 0.;
    
    // initialize scheme for numerically solving the n-body initial value problem
    LeapfrogScheme<DataVariant1, std::vector<Vector3D>> leapfrogScheme(f, dt_, data);

    auto data_new = data;

    // start the time measurement
    NBody_Stopwatch stopwatch;
    stopwatch.start();
    
    // the real computation: do one timestep and increment
    // elapsed simulated time
    while(t<T_)
    {
      // do one timestep
      leapfrogScheme.apply(t, data, data_new);
      data = data_new;

      if (t>=t_out)
      {
        // write output file
        vtkWriter.write(t+dt_,dt_, static_cast<int>(t_out/o_));
        t_out += o_;
      }
      t += dt_;
    }
    // stop time measurement and write measurement results
    stopwatch.stop();
    std::cout << "Total time: " << stopwatch.elapsed() << "s" << std::endl;
  }

private:
  std::size_t N_;
  double T_;
  double dt_;
  double o_;
  std::string init_;
};


#endif // NBODY_HH
