#ifndef LEAPFROGSCHEME_HH
#define LEAPFROGSCHEME_HH

#include <cassert>

#include <iostream>  // provides std::clog

#include "utilities/timefunction.hh"


/**
 * Leapfrog time-stepping scheme for numerically solving the initial value
 * problem
 *   y'' = f(t,y) on [t_0,T], y(t_0) = y_0, y'(t_0) = v_0,
 * with f: [t_0,T] x R^n --> R^n, t_0,T \in R_0^+, y_0,v_0 \in R^n, n \in N.
 * @tparam VectorType The type representing a vector in R^n; it is expected
 *                    to provide an operator[] method for element access and
 *                    a size method like known from STL containers.
 * @tparam TimeType   Type to represent time values.
 */
template <typename Domain, typename Range, typename TimeType = double>
class LeapfrogScheme
{
public:
  typedef TimeFunction<Domain, Range, TimeType> RhsFunction;

  /**
   * Create an instance of the Leapfrog time-stepping scheme.
   * @param[in] f  Right hand side function.
   * @param[in] dt Uniform time step size.
   */
  LeapfrogScheme (const RhsFunction& f, const TimeType& dt, const Domain& y_0)
    : f_(f), dt_(dt), f_new(), f_old(y_0.size())
  {
    f_new.resize(y_0.size());
    f_.evaluate(0., y_0, f_new);

    // print status message in std::clog stream
    std::cout << "Created new instance of LeapfrogScheme with dt = "
              << dt_ << std::endl;
  }

  /**
   * Do one step of Leapfrog time-stepping scheme.
   * @param[in]  t     Start of time step.
   * @param[in]  y_old Value of y at the beginning of time step.
   * @param[out] y_new Value of y at the end of time step.
   * @param[in]  v_old Value of y' at the beginning of time step.
   * @param[out] v_new Value of y' at the end of time step.
   */
  void apply (const TimeType& t, const Domain& y_old, Domain& y_new)
  {
    // evaluate right hand side function f(t,x) at start of time step
    //   f_.evaluate(t_,y_,f_old_);
    // as f_old_ is the same as f_new_ of preceding time step,
    // save this evaluation by swaping f_old_ and f_new_ which in
    // addition can be efficiently executed itself (e.g. in constant
    // time for for an std::vector VectorType)
    std::swap(f_old,f_new);

    // execute first part of Leapfrog method
    for (std::size_t i=0; i<y_old.size(); ++i) {
      assert(y_new[i].r_i.size() == y_old[i].r_i.size());
      assert(y_new[i].r_i.size() == y_old[i].v_i.size());
      y_new[i].r_i = y_old[i].r_i + dt_ * y_old[i].v_i + dt_ * dt_ / 2 * f_old[i];
    }

    // evaluate right hand side function f(t,x) at end of time step
    f_.evaluate(t+dt_, y_new, f_new);

    // execute second part of Leapfrog method
    for (std::size_t i=0; i<y_old.size(); ++i) {
      assert(y_new[i].v_i.size() == y_old[i].v_i.size());
      y_new[i].v_i = y_old[i].v_i + dt_ / 2 * (f_old[i] + f_new[i]);
    }
  }

private:
  const RhsFunction& f_;
  const TimeType& dt_;

  Range f_new;
  Range f_old;
};


#endif // LEAPFROGSCHEME_HH
