#ifndef LEAPFROGSCHEME_HH
#define LEAPFROGSCHEME_HH

#include <cassert>

#include <iostream>  // provides std::clog

#include "utilities/timefunction.hh"


/**
 * Leapfrog time-stepping scheme for numerically solving the initial value
 * problem
 *   y'' = f(t,y) on [t_0,T], y(t_0) = y_0, y'(t_0) = v_0,
 * with f: [t_0,T] x R^n --> R^n, t_0,T \in R_0^+, y_0,v_0 \in R^n, n \in N.
 * @tparam VectorType The type representing a vector in R^n; it is expected
 *                    to provide an operator[] method for element access and
 *                    a size method like known from STL containers.
 * @tparam TimeType   Type to represent time values.
 */
template <typename VectorType, typename TimeType = double>
class LeapfrogScheme
{
public:
  typedef TimeFunction<VectorType,VectorType,TimeType> RhsFunction;

  /**
   * Create an instance of the Leapfrog time-stepping scheme.
   * @param[in] f  Right hand side function.
   * @param[in] dt Uniform time step size.
   */
  LeapfrogScheme (const RhsFunction& f, const TimeType& dt)
    : f_(f), dt_(dt)
  {
    // print status message in std::clog stream
    std::clog << "Created new instance of LeapfrogScheme with dt = "
              << dt_ << std::endl;
  }

  /**
   * Do one step of Leapfrog time-stepping scheme.
   * @param[in]  t     Start of time step.
   * @param[in]  y_old Value of y at the beginning of time step.
   * @param[out] y_new Value of y at the end of time step.
   * @param[in]  v_old Value of y' at the beginning of time step.
   * @param[out] v_new Value of y' at the end of time step.
   */
  void apply (const TimeType& t, /* depends on data layout*/)
  {
    // your code here
  }

private:
  const RhsFunction& f_;
  const TimeType& dt_;
};


#endif // LEAPFROGSCHEME_HH
