//
//  PGMemoryDemo.m
//  Console
//
//  Created by Patrick Förster on 27.03.12.
//  Copyright (c) 2012 __MyCompanyName__. All rights reserved.
//

#import "PGMemoryDemo.h"

@implementation PGMemoryDemo


// 1) Memory Management
//      Wenn Objekte mit alloc erzeugt werden, wird für sie ein Speicher auf dem sogenannten Heap reserviert. 
//      Die Rückgabe einer alloc-Methode ist - wie schon gesehen - vom speziellen Typ "id" und immer ein Zeiger auf den reservierten Speicherplatz.
//      Da Speicherplatz natürlich nur begrenzt vorrätig ist, muss Speicher von Objekten, die nicht mehr benötigt werden, wieder aufgeräumt werden. 
//      Generell gibt es zwei Möglichkeiten wie dies geschehen kann:
//          a) automatische Speicherbereinigung - auch Garbage Collection genannt: Zur Laufzeit wird überprüft, welche Objekte im Speicher noch benötigt werden, 
//              alle anderen werden entfernt
//          b) manuelle Speicherbereinigung: Der Programmierer ist selbst dafür verantwortlich von ihm belegten Speicherplatz, wieder freizugeben, wenn er ihn
//              nicht meht benötigt. "Der Programmierer" ist vielleicht ein wenig beängstigend. Es gibt durchaus Mittel und Wege, auch die manuelle Speicherbereinigung
//              zu automatisieren.
//      Da Apple für iOS-Software immernoch die manuelle Speicherbereinigung vorsieht - obwohl sich dort mit iOS 5 einiges getan - hat, muss man sich zwangsläufig 
//      damit auseinandersetzten. Das Mittel der Wahl heißt hier: "Referenzzähler".
//
//          Der Referenzzähler ist eine Variable, die für jedes Objekt mitzählt, wieviele Referenzen auf dieses Objekt vorliegen. Erst wenn dieser Zähler gleich null ist
//          wird das Objekt nicht mehr benötigt und kann gelöscht werden. Trifft dieser Fall ein, wird automatisch die "dealloc" Methode eines Objektes aufgerufen, damit
//          dieses bevor es gelöscht wird, seinerseits weitere Referenzen freigeben kann.
//
//          Mit dieser Beschreibung werden auch direkt die zwei Gefahren von manueller Speicherbereinigung deutlich:
//              1) Wenn ein Objekt bzw. dessen Speicherplatz freigegeben wird, obwohl dieses noch benutzt wird, ist das Verhalten beim Zugriff auf dieses Objekt undefiniert
//                  (im besten(!) Fall stürzt das Programm ab) (genannt: Overrelease)
//              2) Wenn ein Objekt freigegeben wird, dass noch eine Referenz auf ein anderes Objekt hält, diese aber nicht freigibt, wird der referenzierte Speicherplatz
//                  niemals freigegeben (genannt: Leak)
-(void) run {
    
//  2) Einfaches Beispiel mit retain/release
    [self simpleExamples];
    
//  2b) einen "Leak" forcieren (unendliche While-Schleife.. siehe Methode)
    [self leak];
    
//  3) retain/copy/assign bei Properties
//    [self testAssign];  // wird abstürzen!
    [self testRetain];
    [self testCopy];
    
//  4) autorelease
//      neben "release" existiert noch eine weitere Methode zum dekrementieren des Referenzzählers: "autorelease". Im Gegensatz zu "release" wird bei Aufruf von "autorelease"
//      der Zähler nicht direkt herabgesetzt, sondern erst zu einem (bestimmten) anderen Zeitpunkt. Hierzu werden sogenannte Autorelease-Pools eingesetzt: Sobald an ein Objekt
//      die "autorelease" Nachricht erhält, fügt es sicht selbst zum aktuellen Pool hinzu. Erst wenn der Pool geleert wird, wird für alle in ihm befindlichen Objekte 
//      eine Release-Nachricht abgeschickt und damit der Referenzzähler verringert.
    [self testAutorelease];
    
//      Ein häufiger Fehler:    
//          Viele (Framework-)Klassen bieten sogenannte "Komfortkonstruktoren" als Klassenmethoden.
//          Ganz nach den Regeln liefern diese immer eine Objekt zurück, dass sich bereits in einem Autorelease-Pool befindet. Die Verantworung muss im Gegensatz zum
//          "new" oder "alloc" Konstruktor also explizit übernommen werden!
    
    MemoryObject* object2 = [MemoryObject object];
    MemoryObject* object1 = [MemoryObject new];

    [object1 release]; // richtig
//    [object2 release]; // falsch
    
}

-(void) simpleExamples {
    NSLog(@"Test Retain/Release/retainCount:");

    //  Erzeugung eines Objektes mit "alloc" veranlasst, dass der zugehörige Referenzzähler auf 1 gesetzt wird
    MemoryObject* object = [[MemoryObject alloc] init];
    
    //      Der Referenzzähler eines Objektes ist in Objective-C über [object retainCount] abfragbar. Allerdings ist der zurückgegebene Wert nicht zwangsläufig immer nachvollziehbar  
    //      Bspw. könnte in der Zeit zwischen dem Aufruf von "NSLog" bis zur Ausgabe auf dem Bildschirm eine Referenz freigegeben werden.
    //      Beim Debuggen auf Basis des Referenzzählers ist also vorsicht geboten.
    NSLog(@"Number of references: %lu", [object retainCount]);
    
    //      Im Allgemeinen wird das Referenzen hoch und runterzählen gerne damit verglichen, die Verantwortung für ein Objekt zu übernehmen. Will man das ein Objekt so lange
    //      lebt, wie man es gebenötigt, so muss das Objekt singalisieren, dass es die Verantwortung für das Objekt übernehmen möchte. Solange diese nicht wieder abgeben wird, 
    //      kann das Objekt auch nicht gelöscht werden. Dabei gilt es sich an bestimmte Regeln zu halten:
    //          1) Man hat die Verantwortung für jedes Objekt, das man per "alloc", "new" "copy" oder "mutableCopy" erzeugt
    //          2) Wenn man die Verantworung für ein Objekt übernimmt, muss diese wieder abgegeben werden, wenn das Objekt nicht mehr benötigt wird
    //          2) Niemals die Verantwortung für ein Objekt abgeben, für das man gar nicht verantworlich ist
    
    //      Um den Referenzzähler um eins zu erhöhen und damit zu signalisieren, dass das aufrufende Objekte Interesse an dem referenzierten Objekt hat, wird die Methode
    //      "retain" genutzt
    [object retain];
    NSLog(@"Number of references: %lu", [object retainCount]);
    
    //      Um eine Referenz auf ein Objekt aufzugeben, muss die Methode "release" aufgerufen werden.
    [object release];
    NSLog(@"Number of references: %lu", [object retainCount]);
    
    //      Noch ein "release" aufgrund des vorherigen "retains". Da damit der Referenzzähler auf null fällt, wird automatisch die "dealloc" Methode des Objektes aufgerufen
    [object release];    
    
    NSLog(@"Test Array:");
    
    
    //      Das folgende Beispiel lässt sich am besten im Log nachvollziehen
    object = [[MemoryObject alloc] init];                       // ein Objekt anlegen -> Verantwortung
    NSMutableArray* array = [[NSMutableArray alloc] init];      // ein Objekt anlegen -> Verantwortung
    
    [array addObject:object];                                   // Das Array übernimmt Verantworung
//    [object release];                                           // Verantwortung abgeben
    
    object = [array objectAtIndex:0];
    [object doSomething];
    
    [array release];                                            // Verantwortung abgeben
}

-(void) leak {
    // auskommentieren um den Leak-Monitor zu testen.
/*    while (true) {
        NSMutableArray* array = [[NSMutableArray alloc] init];      // ein Objekt anlegen -> Verantwortung
        MemoryObject* object = [MemoryObject new];
        [array addObject:object];
        [array release];
    }
 */
}

-(void) testAssign {
    NSLog(@"Test Assign:");
    
//  die Klassen-Methode "new" ist ein Synonym für "alloc + init" und liefert ebenfalls eine Referenz auf ein Objekt mit retainCount == 1.
    MemoryObject* object = [MemoryObject new];
    MemoryObject* buddy = [MemoryObject new];
    
//      die Property-Option "assign" weißt der Property lediglich die Referenz zu ohne jegliche Verantwortung zu übernehmen.
    [object setBuddyByAssign:buddy];
    
//      Wird nun der Referenzzähler durch ein anderes Objekt auf null gesenkt.. 
    [buddy release];

//      ..kommt es zu einem Overrelease
    [object.buddy doSomething];
    
//      Man beachte: 
//          NSLog(@"%@", buddy);
//      Würde eine Ausgabe und keinen Fehler erzeugen! Der Speicher auf den "buddy" zeigt, ist zwar bereits freigegeben, beinhaltet aber immernoch die Daten!

    
//      dieser Punkt wird nie erreicht. Würde er sollte hier die Verantwortung für das Objekt wieder abgegeben werden. Da lokal in dieser Methode erzeugt Objekte
//      würde ansonsten niemals seinen Speicherplatz freigeben!
    [object release];    
}

-(void) testRetain {
    NSLog(@"Test Retain:");

    MemoryObject* object = [MemoryObject new];
    MemoryObject* buddy = [MemoryObject new];
    
//      mit der Property-Option "retain" übernimmt "object" durch das Setzen des Buddies die Verantwortung. Der Referenzzähler wird also um eins erhöht.
    [object setBuddyByRetain:buddy];

    NSLog(@"%@ == %@? --> %@", buddy, object.buddy, buddy==object.buddy ? @"YES" : @"NO");

    //      Die Verantwortung in dieser Methode kann nun abgegeben werden.
    [buddy release];
    
//      Der Referenzzähler ist aber durch "retain" immer noch größer 0. Das Buddy-Objekt existiert also noch
    [object.buddy doSomething];
    
//      Die Methode endet, also sollte aufgerufen werden. Im Log sieht man, dass mit dem Release nicht nur "object" dealloziiert wird, sondern auch das zugewiesene 
//      Buddy-Object; "object" gibt seine Verantwortung also regelgerecht wieder ab.
    [object release];
    
//      Was passiert, wenn die Setter-Methode ein weiteres Mal aufgerufen wird?
    NSLog(@"Test Retain II:");
    object = [MemoryObject new];
    
    buddy = [MemoryObject new];
    [object setBuddyByRetain: buddy];
    [buddy release];
    
    buddy = [MemoryObject new];
    [object setBuddyByRetain:buddy];
    [buddy release];
    
    [object release];
}

-(void) testCopy {
    NSLog(@"Test Copy:");
    
    MemoryObject* object = [MemoryObject new];
    MemoryObject* buddy = [MemoryObject new];

//      mit der Property-Option wird nicht die Verantwortung für das übergebene Objekt übernommen, sondern eine Kopie von diesem angelegt und die Verantwortung für diese 
//      übernommen
    [object setBuddyByCopy:buddy];
    
    NSLog(@"%@ == %@? --> %@", buddy, object.buddy, buddy==object.buddy ? @"YES" : @"NO");
    
//      der Referenzzähler wurde durch das Setzen NICHT erhöht. Das Buddy-Object wird also freigegeben.
    [buddy release];
    
/*
    demo aus den folien: 
    MemoryObject* buddy2 = [MemoryObject new];
    [object setBuddyByCopy:buddy2];
    [buddy2 release];
*/
    
//      "object" kann natürlich weiterhin auf seiner Kopie arbeiten...
//    [object.buddy doSomething]; 
    
//      ...und gibt diese bei einem Release auch wieder frei.
    [object release];
}

-(void) testAutorelease {
    NSLog(@"Test Autorelease:");

    MemoryObject* object = [[MemoryObject alloc] init];
    
//  Alle Autorelease-Pools sind in einem Stack angeordnet, d.h. sobald ein neuer erzeugt wird, wird er auf den Stack geschoben und alle Objekte, die ab dann bis zum
//  leeren des Pool eine "autorelease" Nachricht bekommen, werden in diesen Pool "geschoben"
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];    
    
//  Nach den Regeln zur Verantwortlichkeit darf die Verantwortung für Objekt einzig und allein durch Methoden der Art "alloc", "new", "copy" oder "mutableCopy" automatisch
//  auf neu anglegte Objekte übertragen werden. In allen anderen Fällen muss die Verantwortung manuell per "retain" übernommen werden. 
    
    MemoryObject* buddy = [self createMemoryObject];
    
//  Das buddy-Object befindet sich nach den Regeln im Autorelease-Pool und hat einen Referenzzähler von eins. Soll es also das Leeren des Pools überleben, muss
//  es "retained" werden. Hier durch "object".
    [object setBuddyByRetain:buddy];
    
//  Mit "drain" wird der Pool geleert, vom Stack entfernt und sein Referenzzähler dekrementiert. Nach "drain" ist auch der Referenzzähler des Buddy-Objektes dekrementiert worden.
    [pool drain];
    
    /*
     statt den Pool per Hand anzulegen und mit drain zu leeren kann man auch eleganter das Schlüsselwort @autorelease benutzen:
     
     @autorelease {
        MemoryObject* buddy = [self createMemoryObject];
        [object setBuddyByRetain:buddy];
     }
     */
    
    [object.buddy doSomething];
    
//  Aufräumen. 
    [object release];
}


-(MemoryObject*) createMemoryObject {
//  ein neues Objekt wird per "alloc" erzeugt. Diese Methode hat also die Verantwortung für das Objekt.
//  Wird die Verantwortung nicht vor der Rückgabe wieder abgegeben, entsteht ein Speicherleck so denn der Aufrufer nicht über seine neu gewonnene Verantwortung Bescheid weiß.
//  Da diese Methode allerdings nicht der Art "alloc", "new", "copy" oder "mutableCopy" ist, muss der Aufrufer nicht davon ausgehen, durch den Aufruf die Verantwortung übernommen zu haben.
    MemoryObject* object = [[MemoryObject alloc] init];
    
//  Würde man nun hier - ganz Gewissenhaft - seine Verantwortung durch ein "release" wieder abgeben, so würde das Objekt direkt wieder dealloziiert werden! Deshalb wird es 
//  dem aktuellen Autorelease-Pool hinzugefügt. Der Aufrufer kann nun seinerseits mit "retain" die Verantwortung übernehmen, so dass das Objekt das Leeren des Autorelease-Pools
//  überlebt (der Referenzzähler eines Objektes verringert sich pro Aufruf von "autorelease" beim Leeren um eins)
    [object autorelease];
    
    return object;
}

@end


@implementation MemoryObject

+(id) object {
    return [[[MemoryObject alloc] init] autorelease];
}

-(id) init {
    if ((self = [super init])) {
        NSLog(@"%@ has been allocated.", [self description]);

        _message = [[NSString alloc] initWithFormat:@"I'm a MemoryObject called: %@.", [self description]];
    }
    
    return self;
}

-(void) setBuddyByAssign:(MemoryObject *)buddy {
    if (_buddy == buddy) return ;
    
//  einfache Zuweisung durch Kopie der Referenz. Verantwortung wird keine übernommen. 
    _buddy = buddy;
}

-(void) setBuddyByRetain:(MemoryObject *)buddy {
    if (_buddy == buddy) return ;
    
//  falls "_buddy" gesetzt ist, den alten Wert freigeben, bevor der neue gesetzt wird!
    [_buddy release];
    
//  neuen Wert setzten und gleichzeitig die Verantwortung signalisieren
    _buddy = [buddy retain];
}

-(void) setBuddyByCopy:(MemoryObject *)buddy {
    if (_buddy == buddy) return ;

    [_buddy release];
        
    //  es wird nicht der übgebene Wert zu gewiesen, sondern eine Kopie erzeugt. Es muss daher auch keine
    //  Verantwortung für das übergebene Objekt signalisiert werden. Stattdessen wird durch den Aufruf von Copy 
    //  die Verantwortung für die Kopie übernommen. 
    _buddy = [buddy copy];
}

-(MemoryObject*) buddy {
    return _buddy;
}

-(void) doSomething {
    NSLog(@"%@ My retain-count is %lu.", _message, [self retainCount]);
}

-(id )copyWithZone:(NSZone *)zone {
    MemoryObject *copy = [[[self class] allocWithZone: zone] init];
    [copy setBuddyByCopy:_buddy]; 
    return copy;
}

-(id) copy {
    MemoryObject* copy = [[self class] new];

    [copy setBuddyByCopy:_buddy]; 

    return copy;
}

-(id) retain {
    NSLog(@"%@ has been retained.", [self description]);

    return [super retain];
}

-(oneway void) release {
    NSLog(@"%@ has been released.", [self description]);

    [super release];
}
  
-(void) dealloc {
    NSLog(@"%@ has been deallocated.", [self description]);
    
    [_buddy release];
    [_message release];
    
    [super dealloc];
}

@end
